<?php
/*
 * Copyright (c) 2025 AltumCode (https://altumcode.com/)
 *
 * This software is licensed exclusively by AltumCode and is sold only via https://altumcode.com/.
 * Unauthorized distribution, modification, or use of this software without a valid license is not permitted and may be subject to applicable legal actions.
 *
 * 🌍 View all other existing AltumCode projects via https://altumcode.com/
 * 📧 Get in touch for support or general queries via https://altumcode.com/contact
 * 📤 Download the latest version via https://altumcode.com/downloads
 *
 * 🐦 X/Twitter: https://x.com/AltumCode
 * 📘 Facebook: https://facebook.com/altumcode
 * 📸 Instagram: https://instagram.com/altumcode
 */

namespace Altum\Controllers;

use Altum\Alerts;

defined('ALTUMCODE') || die();

class AdminRemovedBackgroundImages extends Controller {

    public function index() {

        /* Prepare the filtering system */
        $filters = (new \Altum\Filters(['user_id', 'project_id'], ['name'], ['removed_background_image_id', 'last_datetime', 'datetime', 'name']));
        $filters->set_default_order_by($this->user->preferences->removed_background_images_default_order_by, $this->user->preferences->default_order_type ?? settings()->main->default_order_type);
        $filters->set_default_results_per_page($this->user->preferences->default_results_per_page ?? settings()->main->default_results_per_page);

        /* Prepare the paginator */
        $total_rows = database()->query("SELECT COUNT(*) AS `total` FROM `removed_background_images` WHERE 1 = 1 {$filters->get_sql_where()}")->fetch_object()->total ?? 0;
        $paginator = (new \Altum\Paginator($total_rows, $filters->get_results_per_page(), $_GET['page'] ?? 1, url('admin/removed-background-images?' . $filters->get_get() . '&page=%d')));

        /* Get the data */
        $removed_background_images = [];
        $removed_background_images_result = database()->query("
            SELECT
                `removed_background_images`.*, `users`.`name` AS `user_name`, `users`.`email` AS `user_email`, `users`.`avatar` AS `user_avatar`
            FROM
                `removed_background_images`
            LEFT JOIN
                `users` ON `removed_background_images`.`user_id` = `users`.`user_id`
            WHERE
                1 = 1
                {$filters->get_sql_where('removed_background_images')}
                {$filters->get_sql_order_by('removed_background_images')}

            {$paginator->get_sql_limit()}
        ");
        while($row = $removed_background_images_result->fetch_object()) {
            $row->settings = json_decode($row->settings ?? '');
            $removed_background_images[] = $row;
        }

        /* Export handler */
        process_export_csv($removed_background_images, ['removed_background_image_id', 'project_id', 'user_id', 'name', 'original_image', 'removed_background_image', 'datetime', 'last_datetime'], sprintf(l('removed_background_images.title')));
        process_export_json($removed_background_images, ['removed_background_image_id', 'project_id', 'user_id', 'name', 'original_image', 'removed_background_image', 'settings', 'datetime', 'last_datetime'], sprintf(l('removed_background_images.title')));

        /* Prepare the pagination view */
        $pagination = (new \Altum\View('partials/admin_pagination', (array) $this))->run(['paginator' => $paginator]);

        /* Main View */
        $data = [
            'removed_background_images' => $removed_background_images,
            'filters' => $filters,
            'pagination' => $pagination,
        ];

        $view = new \Altum\View('admin/removed-background-images/index', (array) $this);

        $this->add_view_content('content', $view->run($data));

    }

    public function bulk() {

        /* Check for any errors */
        if(empty($_POST)) {
            redirect('admin/removed-background-images');
        }

        if(empty($_POST['selected'])) {
            redirect('admin/removed-background-images');
        }

        if(!isset($_POST['type'])) {
            redirect('admin/removed-background-images');
        }

        //ALTUMCODE:DEMO if(DEMO) Alerts::add_error('This command is blocked on the demo.');

        if(!\Altum\Csrf::check()) {
            Alerts::add_error(l('global.error_message.invalid_csrf_token'));
        }

        if(!Alerts::has_field_errors() && !Alerts::has_errors()) {

            set_time_limit(0);

            session_write_close();

            switch($_POST['type']) {
                case 'delete':

                    foreach($_POST['selected'] as $removed_background_image_id) {

                        $image = db()->where('removed_background_image_id', $removed_background_image_id)->getOne('removed_background_images', ['user_id', 'original_image', 'removed_background_image']);

                        /* Delete file */
                        \Altum\Uploads::delete_uploaded_file($image->original_image, 'removed_background_images');
                        \Altum\Uploads::delete_uploaded_file($image->removed_background_image, 'removed_background_images');

                        /* Delete the resource */
                        db()->where('removed_background_image_id', $image->removed_background_image_id)->delete('removed_background_images');

                    }

                    break;
            }

            session_start();

            /* Set a nice success message */
            Alerts::add_success(l('bulk_delete_modal.success_message'));

        }

        redirect('admin/removed-background-images');
    }

    public function delete() {

        $removed_background_image_id = isset($this->params[0]) ? (int) $this->params[0] : null;

        //ALTUMCODE:DEMO if(DEMO) Alerts::add_error('This command is blocked on the demo.');

        if(!\Altum\Csrf::check('global_token')) {
            Alerts::add_error(l('global.error_message.invalid_csrf_token'));
        }

        if(!$image = db()->where('removed_background_image_id', $removed_background_image_id)->getOne('removed_background_images', ['removed_background_image_id', 'user_id', 'name', 'original_image', 'removed_background_image'])) {
            redirect('admin/removed-background-images');
        }

        if(!Alerts::has_field_errors() && !Alerts::has_errors()) {

            /* Delete file */
            \Altum\Uploads::delete_uploaded_file($image->original_image, 'removed_background_images');
            \Altum\Uploads::delete_uploaded_file($image->removed_background_image, 'removed_background_images');

            /* Delete the resource */
            db()->where('removed_background_image_id', $image->removed_background_image_id)->delete('removed_background_images');

            /* Set a nice success message */
            Alerts::add_success(sprintf(l('global.success_message.delete1'), '<strong>' . $image->name . '</strong>'));

        }

        redirect('admin/removed-background-images');
    }

}
